const animals = {
  cat: 'Кот',
  dog: 'Собака',
  bird: 'Птица',
  rodent: 'Грызун'
};

function initDefaultVisits() {
  const saved = localStorage.getItem('vetVisits');
  if (!saved) {
    const defaultVisits = {
      active: [
        {
          id: 1,
          date: '2025-03-25',
          clinic: 'ВетКлиника "Здоровье"',
          diagnosis: 'Профилактический осмотр',
          animal: 'Кот'
        },
        {
          id: 2,
          date: '2025-03-28',
          clinic: 'ВетКлиника "Дружок"',
          diagnosis: 'Вакцинация',
          animal: 'Собака'
        },
        {
          id: 3,
          date: '2025-03-30',
          clinic: 'ВетКлиника "Здоровье"',
          diagnosis: 'Консультация',
          animal: 'Птица'
        }
      ],
      past: [
        {
          id: 4,
          date: '2025-03-15',
          clinic: 'ВетКлиника "Здоровье"',
          diagnosis: 'Профилактический осмотр',
          rating: 5
        },
        {
          id: 5,
          date: '2025-03-10',
          clinic: 'ВетКлиника "Дружок"',
          diagnosis: 'Вакцинация',
          rating: 4
        },
        {
          id: 6,
          date: '2025-03-05',
          clinic: 'ВетКлиника "Здоровье"',
          diagnosis: 'Лечение',
          rating: 5
        }
      ]
    };
    localStorage.setItem('vetVisits', JSON.stringify(defaultVisits));
  }
}

function getVisits() {
  const saved = localStorage.getItem('vetVisits');
  return saved ? JSON.parse(saved) : { active: [], past: [] };
}

function saveVisits(visits) {
  localStorage.setItem('vetVisits', JSON.stringify(visits));
}

function formatDate(dateString) {
  const date = new Date(dateString);
  return date.toLocaleDateString('ru-RU', { day: 'numeric', month: 'long', year: 'numeric' });
}

function initPage() {
  const currentPage = window.location.pathname.split('/').pop();
  
  if (currentPage === 'index.html' || currentPage === '' || currentPage.includes('index')) {
    initVisitsPage();
  } else if (currentPage === 'book.html' || currentPage.includes('book')) {
    initBookingForm();
  }
}

function initVisitsPage() {
  initDefaultVisits();
  const visits = getVisits();
  displayVisits(visits);
  
  setInterval(() => {
    const updated = getVisits();
    displayVisits(updated);
  }, 500);
}

function displayVisits(visits) {
  const container = document.getElementById('visitsContainer');
  if (!container) return;
  
  if (visits.past.length === 0) {
    container.innerHTML = `
      <div class="empty-state">
        <p>У вас пока нет визитов</p>
      </div>
    `;
    return;
  }
  
  container.innerHTML = visits.past.map(visit => `
    <div class="visit-card">
      <div class="visit-header">
        <h3>${visit.clinic}</h3>
      </div>
      <div class="visit-details">
        <div class="detail-item">
          <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"></rect>
            <line x1="16" y1="2" x2="16" y2="6"></line>
            <line x1="8" y1="2" x2="8" y2="6"></line>
            <line x1="3" y1="10" x2="21" y2="10"></line>
          </svg>
          <span>${formatDate(visit.date)}</span>
        </div>
        <div class="detail-item">
          <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"></rect>
            <line x1="9" y1="9" x2="15" y2="9"></line>
            <line x1="9" y1="15" x2="15" y2="15"></line>
          </svg>
          <span>${visit.clinic}</span>
        </div>
        <div class="detail-item">
          <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
            <polyline points="14 2 14 8 20 8"></polyline>
            <line x1="16" y1="13" x2="8" y2="13"></line>
            <line x1="16" y1="17" x2="8" y2="17"></line>
          </svg>
          <span>${visit.diagnosis}</span>
        </div>
      </div>
      <div class="rating-section">
        <p class="rating-label">Оцените визит:</p>
        <div class="rating-stars">
          ${[1, 2, 3, 4, 5].map(star => `
            <button 
              class="star-btn ${visit.rating >= star ? 'filled' : ''}"
              onclick="rateVisit(${visit.id}, ${star})"
            >
              <svg width="24" height="24" viewBox="0 0 24 24" fill="${visit.rating >= star ? 'currentColor' : 'none'}" stroke="currentColor" stroke-width="2">
                <polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"></polygon>
              </svg>
            </button>
          `).join('')}
        </div>
      </div>
    </div>
  `).join('');
}

function rateVisit(visitId, rating) {
  const visits = getVisits();
  const visit = visits.past.find(v => v.id === visitId);
  if (visit) {
    visit.rating = rating;
    saveVisits(visits);
    displayVisits(visits);
  }
}

function initBookingForm() {
  initDefaultVisits();
  
  let selectedAnimal = null;
  let selectedDate = null;
  
  const animalCards = document.querySelectorAll('.animal-card');
  animalCards.forEach(card => {
    card.addEventListener('click', () => {
      animalCards.forEach(c => c.classList.remove('selected'));
      card.classList.add('selected');
      selectedAnimal = card.dataset.animal;
      updateSubmitButton();
    });
  });
  
  const datesGrid = document.getElementById('datesGrid');
  if (datesGrid) {
    const dates = [];
    for (let i = 0; i < 7; i++) {
      const date = new Date();
      date.setDate(date.getDate() + i);
      dates.push(date.toISOString().split('T')[0]);
    }
    
    datesGrid.innerHTML = dates.map(date => {
      const dateObj = new Date(date);
      return `
        <button type="button" class="date-card" data-date="${date}">
          <div class="date-day">${dateObj.getDate()}</div>
          <div class="date-month">${dateObj.toLocaleDateString('ru-RU', { month: 'short' })}</div>
        </button>
      `;
    }).join('');
    
    datesGrid.querySelectorAll('.date-card').forEach(card => {
      card.addEventListener('click', () => {
        datesGrid.querySelectorAll('.date-card').forEach(c => c.classList.remove('selected'));
        card.classList.add('selected');
        selectedDate = card.dataset.date;
        updateSubmitButton();
      });
    });
  }
  
  const submitBtn = document.getElementById('submitBtn');
  
  function updateSubmitButton() {
    if (submitBtn) {
      submitBtn.disabled = !(selectedAnimal && selectedDate);
    }
  }
  
  const form = document.getElementById('bookingForm');
  if (form) {
    form.addEventListener('submit', (e) => {
      e.preventDefault();
      
      if (selectedAnimal && selectedDate) {
        const visits = getVisits();
        const animalName = animals[selectedAnimal];
        
        const newVisit = {
          id: Date.now(),
          date: selectedDate,
          clinic: 'ВетКлиника "Здоровье"',
          diagnosis: 'Запланирован визит',
          animal: animalName,
          status: 'active'
        };
        
        visits.active.push(newVisit);
        saveVisits(visits);
        
        window.location.href = 'index.html';
      }
    });
  }
}

if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initPage);
} else {
  initPage();
}

